function em = importEMachine(dataFile)
arguments
    dataFile string = [];
end

if isempty(dataFile)
    [dataFile, path] = uigetfile({'*.xlsx;*.xls',...
        'Excel Spreadsheet (*.xlsx,*.xls)';
        '*.*',  'All Files (*.*)'}, ...
        'Select a File');
    dataFile = fullfile(path, dataFile);
end
[~, dataFileName] = fileparts(dataFile);

% Main data
data = readtable(dataFile, 'Sheet', 'main', 'ReadRowNames', 1);

em.shortName = dataFileName;
em.inertia = data.value('Moment of inertia'); % inertia coeff, kg*m^2
em.ratedPwr = data.value('Rated power'); % Rated power, kW
em.maxSpd = data.value('Maximum speed') * pi/30; % Maximum Speed, rad/s
em.mass = data.value('Mass'); % kg

% Read full name
tmp = readcell(dataFile,'Sheet','main');
[i,j] = find(strcmp(tmp,'Full Name'));
em.fullName = tmp(i,j+1);

% Torque limit map
opts = detectImportOptions(dataFile,'Sheet','lim');
opts.VariableUnitsRange = 'A2:C2';
limdata = readtable(dataFile, opts);
% Store maxTorque and minTorque interpolants, Nm = f(rad/s)
em.maxTrq = griddedInterpolant(limdata.speed.*pi/30, limdata.maxTorque);
em.minTrq = griddedInterpolant(limdata.speed.*pi/30, limdata.minTorque);

% efficiency map
tmp = readmatrix(dataFile, "Sheet", 'eff');

eta = tmp(2:end,2:end)'./100;
% Efficiencies outside of the operating envelope are filled by linear
% extrapolation to avoid interpolation nonsense near its bounduary
eta(eta==0) = nan;
eta = fillmissing(eta,'linear');
speed = tmp(1,2:end).*pi/30;
torque = tmp(2:end,1);

% Store efficiency map eta = em.etaMap(Speed (rad/s), Torque (Nm))
em.effMap = griddedInterpolant({speed, torque}, eta);

end